/**
 * User Management Service Configuration
 * This file configures the admin frontend to work with external user management service
 */

// Configuration object for external user management service
const USER_MANAGEMENT_CONFIG = {
    // Base URL for user management service
    baseUrl: 'http://localhost:8081/user-management',
    
    // API endpoints
    endpoints: {
        // Authentication
        signin: '/api/auth/signin',
        signup: '/api/auth/signup',
        signout: '/api/auth/signout',
        
        // User Profile Management
        profile: '/api/auth/profile',
        updateProfile: '/api/auth/saveprofile',
        deleteProfile: '/api/auth/userprofile/delete',
        enableUser: '/api/auth/userprofile/enable',
        disableUser: '/api/auth/userprofile/disable',
        
        // Password Management
        forgotPassword: '/api/auth/forgot-password',
        resetPassword: '/api/auth/reset-password',
        
        // Address Management
        addresses: '/api/auth/address',
        saveAddress: '/api/auth/saveaddress',
        
        // Contact Management
        contacts: '/api/auth/contact',
        saveContact: '/api/auth/savecontact',
        
        // Metadata
        metadata: '/api/auth/metadata',
        locations: '/api/auth/locations'
    },
    
    // JWT token storage key
    tokenKey: 'userManagementToken',
    
    // Request headers
    headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
    }
};

/**
 * User Management API Service
 */
class UserManagementService {
    constructor(config) {
        this.config = config;
        this.token = localStorage.getItem(config.tokenKey);
    }
    
    /**
     * Get full URL for an endpoint
     */
    getUrl(endpoint) {
        return this.config.baseUrl + this.config.endpoints[endpoint];
    }
    
    /**
     * Get headers with authentication token
     */
    getHeaders() {
        const headers = { ...this.config.headers };
        if (this.token) {
            headers['Authorization'] = `Bearer ${this.token}`;
        }
        return headers;
    }
    
    /**
     * Make API request
     */
    async makeRequest(endpoint, method = 'GET', data = null) {
        const url = this.getUrl(endpoint);
        const options = {
            method: method,
            headers: this.getHeaders()
        };
        
        if (data && (method === 'POST' || method === 'PUT')) {
            options.body = JSON.stringify(data);
        }
        
        try {
            const response = await fetch(url, options);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            return await response.json();
        } catch (error) {
            console.error('API request failed:', error);
            throw error;
        }
    }
    
    /**
     * Authentication methods
     */
    async signin(credentials) {
        const response = await this.makeRequest('signin', 'POST', credentials);
        if (response.token) {
            this.token = response.token;
            localStorage.setItem(this.config.tokenKey, this.token);
        }
        return response;
    }
    
    async signup(userData) {
        return await this.makeRequest('signup', 'POST', userData);
    }
    
    async signout() {
        const response = await this.makeRequest('signout', 'POST');
        this.token = null;
        localStorage.removeItem(this.config.tokenKey);
        return response;
    }
    
    /**
     * User profile methods
     */
    async getAllUsers() {
        return await this.makeRequest('profile', 'GET');
    }
    
    async updateProfile(profileData) {
        return await this.makeRequest('updateProfile', 'POST', profileData);
    }
    
    async deleteUser(userId) {
        return await this.makeRequest('deleteProfile', 'DELETE', { id: userId });
    }
    
    async enableUser(userId) {
        return await this.makeRequest('enableUser', 'POST', { id: userId });
    }
    
    async disableUser(userId) {
        return await this.makeRequest('disableUser', 'POST', { id: userId });
    }
    
    /**
     * Password management
     */
    async forgotPassword(email) {
        return await this.makeRequest('forgotPassword', 'POST', { email });
    }
    
    async resetPassword(token, newPassword) {
        return await this.makeRequest('resetPassword', 'POST', { token, password: newPassword });
    }
    
    /**
     * Metadata methods
     */
    async getMetadata() {
        return await this.makeRequest('metadata', 'GET');
    }
    
    async getLocations() {
        return await this.makeRequest('locations', 'GET');
    }
}

// Initialize the service
const userManagementService = new UserManagementService(USER_MANAGEMENT_CONFIG);

// Make it globally available
window.UserManagementService = userManagementService;
window.USER_MANAGEMENT_CONFIG = USER_MANAGEMENT_CONFIG;

/**
 * Utility functions for admin interface
 */
const AdminUtils = {
    /**
     * Show success message
     */
    showSuccess(message) {
        // Using SmartAdmin notification system
        if (typeof $.smallBox === 'function') {
            $.smallBox({
                title: "Success",
                content: message,
                color: "#659265",
                iconSmall: "fa fa-check fa-2x fadeInRight animated",
                timeout: 4000
            });
        } else {
            alert('Success: ' + message);
        }
    },
    
    /**
     * Show error message
     */
    showError(message) {
        if (typeof $.smallBox === 'function') {
            $.smallBox({
                title: "Error",
                content: message,
                color: "#C46A69",
                iconSmall: "fa fa-warning fa-2x fadeInRight animated",
                timeout: 4000
            });
        } else {
            alert('Error: ' + message);
        }
    },
    
    /**
     * Check if user is authenticated
     */
    isAuthenticated() {
        return !!localStorage.getItem(USER_MANAGEMENT_CONFIG.tokenKey);
    },
    
    /**
     * Redirect to login if not authenticated
     */
    requireAuth() {
        if (!this.isAuthenticated()) {
            window.location.href = '/admin/login.html';
            return false;
        }
        return true;
    }
};

// Make AdminUtils globally available
window.AdminUtils = AdminUtils;

console.log('User Management Service configured for admin interface');
