/**
 * Professional Component Loader with Caching
 * Prevents unnecessary reloading and flickering of header/sidebar
 * Version: 1.0.0
 */

class ComponentLoader {
  constructor() {
    this.cache = new Map();
    this.loadingPromises = new Map();
  }

  /**
   * Load a component with caching support
   * @param {string} componentPath - Path to the component HTML file
   * @param {string} targetSelector - jQuery selector for target element
   * @param {Function} callback - Optional callback after component loads
   * @returns {Promise}
   */
  async load(componentPath, targetSelector, callback) {
    try {
      // Check if already loading this component
      if (this.loadingPromises.has(componentPath)) {
        const html = await this.loadingPromises.get(componentPath);
        this._insertComponent(targetSelector, html, callback);
        return html;
      }

      // Check cache first
      if (this.cache.has(componentPath)) {
        const html = this.cache.get(componentPath);
        this._insertComponent(targetSelector, html, callback);
        return html;
      }

      // Load component from server
      const loadPromise = this._fetchComponent(componentPath);
      this.loadingPromises.set(componentPath, loadPromise);

      const html = await loadPromise;
      
      // Cache the component
      this.cache.set(componentPath, html);
      this.loadingPromises.delete(componentPath);

      // Insert into DOM
      this._insertComponent(targetSelector, html, callback);

      return html;
    } catch (error) {
      console.error(`Failed to load component: ${componentPath}`, error);
      this.loadingPromises.delete(componentPath);
      throw error;
    }
  }

  /**
   * Fetch component HTML from server
   * @private
   */
  async _fetchComponent(componentPath) {
    const response = await fetch(componentPath);
    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }
    return await response.text();
  }

  /**
   * Insert component into DOM
   * @private
   */
  _insertComponent(targetSelector, html, callback) {
    const $target = $(targetSelector);
    
    if ($target.length === 0) {
      console.warn(`Target element not found: ${targetSelector}`);
      return;
    }

    // Check if component is already loaded with actual content
    // Look for key elements that indicate the component is loaded
    const isHeaderLoaded = targetSelector === '#header' && $target.find('.page-header, header.page-header').length > 0;
    const isSidebarLoaded = targetSelector === '#sidebar' && $target.find('.page-sidebar, aside.page-sidebar').length > 0;
    
    if (isHeaderLoaded || isSidebarLoaded) {
      // Component already loaded, just trigger callback to update user info
      console.log(`✓ ${targetSelector} already loaded, updating data only`);
      if (callback && typeof callback === 'function') {
        callback();
      }
      return;
    }

    // Only insert if not already loaded
    if ($target.children().length === 0 || (!isHeaderLoaded && !isSidebarLoaded)) {
      $target.html(html);
      console.log(`✓ ${targetSelector} loaded fresh`);
    }

    // Execute callback
    if (callback && typeof callback === 'function') {
      callback();
    }
  }

  /**
   * Clear cache for specific component or all components
   * @param {string} componentPath - Optional path to clear specific component
   */
  clearCache(componentPath = null) {
    if (componentPath) {
      this.cache.delete(componentPath);
    } else {
      this.cache.clear();
    }
  }

  /**
   * Reload a component (bypasses cache)
   * @param {string} componentPath - Path to the component HTML file
   * @param {string} targetSelector - jQuery selector for target element
   * @param {Function} callback - Optional callback after component loads
   */
  async reload(componentPath, targetSelector, callback) {
    this.clearCache(componentPath);
    return await this.load(componentPath, targetSelector, callback);
  }

  /**
   * Preload components for faster page loads
   * @param {Array<string>} componentPaths - Array of component paths to preload
   */
  async preload(componentPaths) {
    const promises = componentPaths.map(path => this._fetchComponent(path));
    const results = await Promise.all(promises);
    
    componentPaths.forEach((path, index) => {
      this.cache.set(path, results[index]);
    });

    console.log(`Preloaded ${componentPaths.length} components`);
  }
}

// Create global instance
window.componentLoader = new ComponentLoader();

/**
 * Enhanced load user info for header and sidebar
 * @param {object} userInfo - User information object
 */
function initializeUserInfo(userInfo) {
  // Trigger userInfo event for header
  $("#header").trigger("userInfo", userInfo);
  
  // Trigger userInfo event for sidebar
  $("#sidebar").trigger("userInfo", userInfo);
}

/**
 * Load page components (header + sidebar) with proper panel detection
 * @param {object} userInfo - User information object
 * @param {string} panel - Panel type: 'rental' or 'property-list'
 */
async function loadPageComponents(userInfo, panel = null) {
  try {
    // Determine panel from localStorage if not provided
    if (!panel) {
      panel = localStorage.getItem("panel") || "rental";
    }

    // Check if we need to switch panels (sidebar change required)
    const lastLoadedPanel = window._lastLoadedPanel;
    const panelChanged = lastLoadedPanel && lastLoadedPanel !== panel;
    
    // Store current panel
    window._lastLoadedPanel = panel;

    // Determine paths based on current page location
    const currentPath = window.location.pathname;
    let headerPath, sidebarPath;

    if (currentPath.includes("/admin/rental/") || currentPath.includes("/admin/property-list/")) {
      // For pages in rental or property-list subdirectories
      if (currentPath.includes("/property-management/")) {
        // For nested property-management pages
        headerPath = "../../../components/header.html";
        sidebarPath = `../../../components/${panel === "property-list" ? "propertylist" : "rental"}-sidebar.html`;
      } else {
        // For dashboard pages
        headerPath = "../../components/header.html";
        sidebarPath = `../../components/${panel === "property-list" ? "propertylist" : "rental"}-sidebar.html`;
      }
    } else if (currentPath.includes("/admin/user-management/")) {
      // For user management pages
      headerPath = "../../components/header.html";
      sidebarPath = `../../components/${panel === "property-list" ? "propertylist" : "rental"}-sidebar.html`;
    } else if (currentPath.includes("/admin/organization-management/")) {
      // For organization management pages
      headerPath = "../../components/header.html";
      sidebarPath = `../../components/${panel === "property-list" ? "propertylist" : "rental"}-sidebar.html`;
    }

    // Check if components are already in DOM
    const $header = $("#header");
    const $sidebar = $("#sidebar");
    const headerExists = $header.find('.page-header, header.page-header').length > 0;
    const sidebarExists = $sidebar.find('.page-sidebar, aside.page-sidebar').length > 0;

    // Load header only if not exists
    if (!headerExists) {
      await componentLoader.load(headerPath, "#header", function() {
        $("#header").trigger("userInfo", userInfo);
      });
    } else {
      // Just update user info
      $("#header").trigger("userInfo", userInfo);
      console.log("✓ Header already present, updating data only");
    }

    // Load sidebar only if not exists OR panel changed
    if (!sidebarExists || panelChanged) {
      if (panelChanged) {
        // Clear sidebar cache and force reload for panel switch
        componentLoader.clearCache(sidebarPath);
        $sidebar.empty(); // Clear existing sidebar
        console.log(`🔄 Panel switched to ${panel}, reloading sidebar`);
      }
      
      await componentLoader.load(sidebarPath, "#sidebar", function() {
        $("#sidebar").trigger("userInfo", userInfo);
      });
    } else {
      // Just update user info
      $("#sidebar").trigger("userInfo", userInfo);
      console.log("✓ Sidebar already present, updating data only");
    }

    console.log("✅ Page components ready");
  } catch (error) {
    console.error("❌ Error loading page components:", error);
  }
}

// Export functions
window.loadPageComponents = loadPageComponents;
window.initializeUserInfo = initializeUserInfo;
